/*
 * Decompiled with CFR 0.152.
 */
package uniprot;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

public class UniprotMappingParser {
    private final Map<String, Integer> reviewedEntries = new HashMap<String, Integer>();
    private final String mappingFileName;
    private boolean isOnlyReviewed = true;

    public UniprotMappingParser(String mappingFileName) {
        this.mappingFileName = mappingFileName;
    }

    private static Map<String, Integer> mapArrayElementToIndex(List<String> arr) {
        HashMap<String, Integer> elementToIndex = new HashMap<String, Integer>();
        int len = arr.size();
        for (int i = 0; i < len; ++i) {
            elementToIndex.put(arr.get(i), i);
        }
        return elementToIndex;
    }

    private static <T> Map<T, Boolean> mapSetElementToBoolean(Set<T> elements, boolean bValue) {
        HashMap<T, Boolean> map = new HashMap<T, Boolean>();
        for (T el : elements) {
            map.put(el, bValue);
        }
        return map;
    }

    private static boolean isFileExists(String fileName) {
        File file = new File(fileName);
        if (!file.isFile()) {
            return false;
        }
        return file.exists();
    }

    private static void displayHelp() {
        System.out.println("Description:");
        System.out.println("UniprotMappingParser is a simple tool that converts one set of identifiers to another one.\nThe tool is inspired by UniProt mapping tool (https://www.uniprot.org/uploadlists/) .\n");
        System.out.println("Usage:");
        System.out.println("java -cp DiNGO.jar uniprot.UniprotMappingParser -i <iFile> -f <from> -t <to> -m <uniprot mapping file> [-s swiss-prot file] [-n]\n");
        System.out.println("Options:");
        System.out.printf("%-30s%s%n", "    -i <iFile>", "Input file name containing list of IDs\n");
        System.out.printf("%-30s%s%n", "    -f <from>", "input ID\n");
        System.out.printf("%-30s%s%n", "    -t <to>", "output ID\n");
        System.out.printf("%-30s%s%n", "    -m <uniprot mapping file>", "UniProt idmapping.dat file\n");
        System.out.printf("%-30s%s%n", "    -s swiss-prot file", "uniprot.sprot.fasta file\n");
        System.out.printf("%-30s%s%n", "    -n", "uses non-reviewed UniProt entries\n");
        System.out.println("An example: ");
        System.out.println("java -cp Dingo.jar uniprot.UniprotMappingParser -i input.txt -t UniProtKB -f Gene_Name -m HUMAN_9606_idmapping.dat -s uniprot_sprot.fasta\n");
        System.out.println("HUMAN_9606_idmapping.dat and uniprot_sprot.fasta files can be downloaded from ftp://ftp.uniprot.org/pub/databases/uniprot/current_release/knowledgebase/\n");
        System.out.println("Description of IDs:\n");
        System.out.printf("%-20s%s%n", "ID", "Description");
        System.out.println("-------------------------------------------------------------------------------");
        System.out.printf("%-20s%s%n", "UniProtKB", "UniProt identification (Q8IZP0 and P30457)");
        System.out.printf("%-20s%s%n", "UniProtKB-ID", "UniProt identification (1433B_HUMAN)");
        System.out.printf("%-20s%s%n", "Gene_Name", "Gene symbol (PTEN or CDKN2A)");
        System.out.printf("%-20s%s%n", "GeneID", "Entrez identification (7531)");
        System.out.printf("%-20s%s%n", "Ensembl", "Ensembl gene identification (ENSG00000108953)");
        System.out.printf("%-20s%s%n", "Ensembl_TRS", "Ensembl transcript identification (ENST00000264335)");
        System.out.printf("%-20s%s%n", "Ensembl_PRO", "Ensembl protein identification (ENSP00000264335)");
        System.out.printf("%-20s%s%n", "UniGene", "UniGene identification (Hs.643544)");
        System.out.println("-------------------------------------------------------------------------------\n");
        System.out.println("For full description look at ftp://ftp.uniprot.org/pub/databases/uniprot/current_release/knowledgebase/idmapping/README");
    }

    public boolean extractCuratedUniProtEntries(String swissProtFile) {
        try (BufferedReader reader = new BufferedReader(new FileReader(swissProtFile));){
            String line;
            while ((line = reader.readLine()) != null) {
                if (!line.startsWith(">sp")) continue;
                int first = line.indexOf("|");
                int last = line.lastIndexOf("|");
                String uniProtKB = line.substring(first + 1, last);
                this.reviewedEntries.put(uniProtKB, 1);
            }
        }
        catch (IOException ex) {
            Logger.getLogger(UniprotMappingParser.class.getName()).log(Level.INFO, swissProtFile + " not found!", "Results will include TrEMBL and Swiss-Prot entries");
            return false;
        }
        return true;
    }

    public void mapSelectedIDsInTabFormat(String outFile, String ... ids) {
        try (BufferedReader reader = new BufferedReader(new FileReader(this.mappingFileName));
             BufferedWriter writer = new BufferedWriter(new FileWriter(outFile));){
            String line;
            if (ids == null || ids.length == 0) {
                System.out.println("Please provide list of IDs!");
                return;
            }
            String protein = "";
            boolean isFirst = true;
            List<String> targets = Arrays.asList(ids);
            Map<String, Integer> map = UniprotMappingParser.mapArrayElementToIndex(targets);
            String[] hits = new String[ids.length];
            while ((line = reader.readLine()) != null) {
                String id;
                int fPos = line.indexOf("\t");
                int lPos = line.lastIndexOf("\t");
                if (line.substring(0, fPos).contains("-")) continue;
                if (!protein.equals("") && !protein.equals(line.substring(0, fPos))) {
                    writer.write(protein);
                    for (String e : hits) {
                        if (e != null) {
                            writer.write(e);
                            continue;
                        }
                        writer.write("");
                    }
                    writer.newLine();
                    hits = new String[ids.length];
                    protein = line.substring(0, fPos);
                }
                if (isFirst) {
                    protein = line.substring(0, fPos);
                    isFirst = false;
                }
                if (!targets.contains(id = line.substring(fPos + 1, lPos))) continue;
                String element = hits[map.get(id)];
                if (element == null) {
                    hits[map.get((Object)id).intValue()] = line.substring(lPos);
                    continue;
                }
                hits[map.get((Object)id).intValue()] = element.substring(0, element.length() - 1) + ", " + line.substring(lPos).substring(1);
            }
        }
        catch (IOException ex) {
            System.out.println(ex.getMessage());
        }
    }

    public void mappingIDs(Set<String> inputIDs, String from, String to, String swissProtFile) {
        if (inputIDs == null || inputIDs.isEmpty()) {
            System.out.println("Please provide list of IDs!");
            return;
        }
        if (!(from.equals("UniProtKB") || to.equals("UniProtKB") || from.equals("UniProtKB-ID") || to.equals("UniProtKB-ID"))) {
            System.out.println("Can not do mapping from " + from + " to " + to);
            return;
        }
        try (BufferedReader reader = new BufferedReader(new FileReader(this.mappingFileName));
             BufferedWriter writer = new BufferedWriter(new FileWriter("mapping.tab"));){
            String line;
            HashMap<String, String> uniToUniID = new HashMap<String, String>();
            Map<String, Boolean> map = UniprotMappingParser.mapSetElementToBoolean(inputIDs, false);
            String id = "";
            boolean bValue = this.extractCuratedUniProtEntries(swissProtFile);
            while ((line = reader.readLine()) != null) {
                int lPos;
                int fPos = line.indexOf("\t");
                id = line.substring(fPos + 1, lPos = line.lastIndexOf("\t"));
                if (id.equals("Gene_Synonym")) {
                    id = "Gene_Name";
                }
                String firstColumn = line.substring(0, fPos);
                String thirdColumn = line.substring(lPos + 1);
                if (this.isOnlyReviewed && bValue && !this.reviewedEntries.containsKey(firstColumn)) continue;
                if (from.equals("UniProtKB")) {
                    if (!id.equals(to) || !inputIDs.contains(firstColumn)) continue;
                    writer.write(firstColumn + line.substring(lPos));
                    writer.newLine();
                    map.put(firstColumn, true);
                    continue;
                }
                if (from.equals("UniProtKB-ID")) {
                    if (id.equals("UniProtKB-ID")) {
                        uniToUniID.put(firstColumn, thirdColumn);
                    }
                    if (id.equals(to) && inputIDs.contains(uniToUniID.get(firstColumn))) {
                        writer.write((String)uniToUniID.get(firstColumn) + "\t" + thirdColumn);
                        writer.newLine();
                        map.put((String)uniToUniID.get(firstColumn), true);
                    }
                    if (!to.equals("UniProtKB") || !inputIDs.contains(line.substring(lPos + 1))) continue;
                    writer.write(thirdColumn + "\t" + firstColumn);
                    writer.newLine();
                    map.put(thirdColumn, true);
                    continue;
                }
                if (to.equalsIgnoreCase("UniProtKB-ID") && id.equals("UniProtKB-ID")) {
                    uniToUniID.put(firstColumn, thirdColumn);
                }
                if (!id.equals(from) || !inputIDs.contains(thirdColumn)) continue;
                if (to.equals("UniProtKB-ID")) {
                    writer.write(thirdColumn + "\t" + (String)uniToUniID.get(firstColumn));
                } else {
                    writer.write(thirdColumn + "\t" + firstColumn);
                }
                writer.newLine();
                map.put(thirdColumn, true);
            }
            System.out.println("\nResults have been saved to mapping.tab file!\n");
            System.out.println("Not mapped entries: ");
            for (Map.Entry<String, Boolean> entry : map.entrySet()) {
                if (entry.getValue().booleanValue()) continue;
                System.out.println(entry.getKey());
            }
        }
        catch (IOException ex) {
            System.out.println(ex.getMessage());
        }
    }

    public boolean isIsOnlyReviewed() {
        return this.isOnlyReviewed;
    }

    public void setIsOnlyReviewed(boolean isOnlyReviewed) {
        this.isOnlyReviewed = isOnlyReviewed;
    }

    public static void main(String[] args) {
        if (args.length < 4) {
            UniprotMappingParser.displayHelp();
            return;
        }
        String fileName = "";
        String uniprotMapingFile = "";
        String from = "";
        String to = "";
        String swissProt = "";
        boolean bReviewed = true;
        int ln = args.length;
        for (int i = 0; i < ln; ++i) {
            if (args[i].equals("-i")) {
                if (i + 1 >= ln) continue;
                fileName = args[i + 1];
                continue;
            }
            if (args[i].equals("-f")) {
                if (i + 1 >= ln) continue;
                from = args[i + 1];
                continue;
            }
            if (args[i].equals("-t")) {
                if (i + 1 >= ln) continue;
                to = args[i + 1];
                continue;
            }
            if (args[i].equals("-m")) {
                if (i + 1 >= ln) continue;
                uniprotMapingFile = args[i + 1];
                continue;
            }
            if (args[i].equals("-s")) {
                if (i + 1 >= ln) continue;
                swissProt = args[i + 1];
                continue;
            }
            if (!args[i].equals("-n")) continue;
            bReviewed = false;
        }
        if (!UniprotMappingParser.isFileExists(fileName)) {
            System.out.println("File " + fileName + " has not been found!");
            return;
        }
        if (!UniprotMappingParser.isFileExists(uniprotMapingFile)) {
            System.out.println("Mapping file " + uniprotMapingFile + " does not exist!");
            return;
        }
        LinkedHashSet<String> input = new LinkedHashSet<String>();
        try (BufferedReader reader = new BufferedReader(new FileReader(fileName));){
            String line;
            while ((line = reader.readLine()) != null) {
                input.add(line.trim());
            }
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
            return;
        }
        UniprotMappingParser test = new UniprotMappingParser(uniprotMapingFile);
        test.setIsOnlyReviewed(bReviewed);
        test.mappingIDs(input, from, to, swissProt);
    }
}

